//*************************************************************************************************
//
//	Description:
//		lightglow_billboard.fx - Billboarded shader for use with vehicle light glows.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     18/07/2008  0.1           Created
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

// The maximum light ID the shader supports (important when calculating visibility U coord)
#define MAX_LIGHT_ID	32.0f

// The speed of rotations; vary this to speed them up or slow them down
#define ROTATION_SPEED 0.75f;


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;

float4x4 viewI : ViewI
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif



//
// Transforms
//
#if defined( _3DSMAX_ ) || defined(_PS3_)
// Max doesn't support viewproj as an app-supplied parameter
SHARE_PARAM float4x4 worldview : WorldView
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

SHARE_PARAM float4x4 view : View
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = false;
>;

SHARE_PARAM float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

SHARE_PARAM float4x4 projMatrix : Projection
<
	bool appEdit = false;
	bool export = false;
>;
#else


float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

SHARE_PARAM float4x4 view : View
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = false;
>;


SHARE_PARAM float4x4 projMatrix : Projection
<
	bool appEdit = false;
	bool export = false;
	bool dynamic = false;
>;
#endif


//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_
// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Second UV holds the billboarding pivot offsets
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
	bool use3dMapping = true;
> = 0;

// Vertex colour channel
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Used for the emissive lighting control
int texcoord4 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 4;
	int MapChannel = 3;
	int RuntimeTexcoord = 2;
	bool ProtectFromRescaling = true;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Textures
//

SPECIALISATION_PARAM( tracksideLight, "Trackside Light?", "TRACKSIDE_LIGHT" )	// TRUE if this is a trackside light (so it gets its visibility coords from params)

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

// Used to apply distance scaling (by default glows are the same size always.
// in practise we like to scale them down a bit with distance, 
float distanceScale
<
	string UIName = "Distance Scale";
	bool appEdit = true;
	bool export = true;
> = 1.0f;


#define NUM_EMISSION_CONTROL_VALUES		20
float emissionControl[ NUM_EMISSION_CONTROL_VALUES ]
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
>;

#if defined( _3DSMAX_ )
float testLightLevel
<
	string UIName = "Test Light Level";
	bool appEdit = false;
	bool export = false;
> = 0.0f;
#endif

#if !defined( _3DSMAX_ )
texture visibilityTexture : TEXTURE							// Visibility results texture
<
	bool appEdit = true;
>;

#if defined( TRACKSIDE_LIGHT )
float occlusionMapUCoord
<
	bool appEdit = true;
> = 0.0f;
#endif	// defined( TRACKSIDE_LIGHT )

float occlusionMapVCoord
<
	bool appEdit = true;
> = 0.0f;
#endif	// !defined( _3DSMAX_ )


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


#if !defined( _3DSMAX_ )
sampler2D visibilityMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="visibilityTexture"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < visibilityTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif



//
// Functions
//

// Function calculates emissiveness by looking up a value from an array which the app sets,
// indexed by the control ID passed. If a negative ID is passed, the emission is considered to
// be always on.

float CalcEmissiveness( float _ID, float _brightness )
{
	int intControlID = floor( _ID );

	if ( intControlID >= 0 )
	{
#if defined( _3DSMAX_ )
		return testLightLevel * abs( _brightness );
#else
		return min( emissionControl[ intControlID ] * abs( _brightness ), 2.0f );
#endif
	}
	else
	{
		return 0.0f;
	}
}


float3 DoArbitraryBillboardingWithScale( float3 _pivotLocalPos, float3 _vertexToPivot, float _scaleFactor, float4x4 _worldview )
{
	float3 result = mul( float4( _pivotLocalPos, 1.0f ), _worldview ).xyz;

	_vertexToPivot.z = -_vertexToPivot.z;
	result += ( _vertexToPivot.xyz * _scaleFactor );

	return result;
}




//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
//
// Beware that input structure is hardcoded in VehicleRender, too.
// We analyse the mesh, so do not change the input stream - neither sizes nor semantics.
//
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
	float3 billboardPivot: TEXCOORD1;											// Pivot-relative vertex position.
	float3 colour   : TEXCOORD2;													// Vertex colour																											// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD3;													// Vertex alpha
	float2 lightControlValues : TEXCOORD4;								// UV channel 5 texture coord (emissive control)
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
	float3 billboardPivot: TEXCOORD1;											// Pivot-relative vertex position, with Y and Z swapped
	float2 lightControlValues : TEXCOORD2;								// UV channel 5 texture coord (emissive control)
#endif
	float3 normal   : NORMAL;															// Object space normal
};



// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD3;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
	float2 visTexCoord : TEXCOORD1;												// UV coords for visibility texture
	float emissiveness : TEXCOORD2;												// Emissiveness amount
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//



VSOUTPUT LightglowVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4	worldview = mul( world, view );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	_output.emissiveness = max( CalcEmissiveness( _input.lightControlValues.x, _input.lightControlValues.y ), 0.0f );

	// Calculate the visibility texture coord
#if defined( _3DSMAX_ )
	_output.visTexCoord = float2( 0.0f, 0.0f );
#else

#if defined( TRACKSIDE_LIGHT )
	// Trackside lights get their U coord passed as a shader param
	_output.visTexCoord = float2( occlusionMapUCoord, occlusionMapVCoord );
#else
	// Vehicle lights use a consistent formula for the U coord
	_output.visTexCoord = float2( ( _input.lightControlValues.x + 0.5f ) / MAX_LIGHT_ID, occlusionMapVCoord );
#endif

#endif

	// Calculate the pivot position in local coords
	float3 billboardPivot = _input.billboardPivot;
#if defined( _3DSMAX_ )
	billboardPivot.y += 1.0f;
	float3 localPivot = _input.position - billboardPivot;
#else
	float3 localPivot = _input.position - billboardPivot.xzy;
#endif

	// View-project the pivot
	float3 viewPivot = mul( float4( localPivot, 1.0f ), worldview ).xyz;
	float dist = length( viewPivot );

	// Scale the apparent distance to compensate for the field of view (1.0393 is a standard 60 degrees)
	float FOVadjustment = ( 1.0f / projMatrix[0].x ) / tan( 1.0393f );
	dist *= FOVadjustment;

	// Calculate view coords pos
#if defined( _3DSMAX_ )
	float scaleFactor = 1.0f;
#else
	float scaleFactor = pow( dist, 0.3f);

#endif
	float3 viewPos = DoArbitraryBillboardingWithScale( localPivot, billboardPivot, scaleFactor, worldview );

	// Calculate the rotation around the pivot, apply it to the delta, and re-offset the vertex from the pivot
	float3 viewDelta = viewPos - viewPivot;
	float4 projectedPivot = mul( float4( viewPivot, 1.0f ), projMatrix );
	float4 NDCProjectedPivot = projectedPivot / projectedPivot.w;

	float angle = NDCProjectedPivot.x * ROTATION_SPEED;
	float s = sin(angle);
	float c = cos(angle);
	float3 rotatedDelta = float3( ( c * viewDelta.x ) + ( s * viewDelta.y ),
																( -s * viewDelta.x ) + ( c * viewDelta.y ),
																0.0f );
	viewPos = viewPivot + rotatedDelta;

	// Project the vertex into clip space and push it onto the nearplane
	float4 projectedPos = mul( float4( viewPos, 1.0f ), projMatrix );
	float4 NDCProjectedPos = projectedPos / projectedPos.w;
	NDCProjectedPos.z = 0.0f;
	_output.position = NDCProjectedPos;

	// Transform object-space normal into view coords
	float3 viewNormal = mul( float4( _input.normal, 0.0f ), worldview ).xyz;

	// Calculate angle opacity falloff (based on dot with the eye vector squared)
	float3 viewEyeVec = -normalize( viewPivot );
	float normDotView = saturate( dot( viewNormal, viewEyeVec ) );
	_output.colour.a *= normDotView * normDotView;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

struct PSINPUT
{
	float4 colour		: TEXCOORD3;								// Vertex colour
	float2 texCoord	: TEXCOORD0;						// UV coords for texture channel 0
	float2 visTexCoord : TEXCOORD1;					// UV coords for visibility texture
	float emissiveness : TEXCOORD2;					// Emissiveness amount
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT LightglowFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
#if defined( _3DSMAX_ )
	float visibilityResult = 1.0f;
#else
	float visibilityResult = tex2D( visibilityMap, _input.visTexCoord ).r;
#endif

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Apply emissiveness value
	accumulatedColour.rgb *= _input.emissiveness;

	// Apply visibility value
	accumulatedColour.rgb *= visibilityResult;

	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Lightglow
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = false;
	
	bool isBillboard = true;
	
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Lightglow";
	int    normalDeferredID		= 2;
	
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = false;
		bool ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "ONE";
		string BlendOp = "ADD";
	>
#endif
	{
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
#ifdef _PS3_
		BlendFunc=int2(SrcAlpha, One);
		BlendEquation=int(FuncAdd);
#else
		SrcBlend = SRCALPHA;
		DestBlend = ONE;
		BlendOp = ADD;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx LightglowVertexShader();
		PixelShader = compile sce_fp_rsx LightglowFragmentShader();
#else
		VertexShader = compile vs_3_0 LightglowVertexShader();
		PixelShader = compile ps_3_0 LightglowFragmentShader();
#endif
	}
}
